#!/bin/bash

. '/etc/bat60_scripts/scripts/includes/bat60_script_globals.sh'

LOG_FPATH="$LOG_DIR"/"$LOG_FNAME"

###############################################################################
#	@brief		Beendet alle Daemons und entfernt ihre lock-Files, wenn diese existieren
#
function daemons_off
{
	#	Daemons beenden
	bla=$(bat60_wtchdgd -s 2>/dev/null)
	bla=$(killall -KILL bat60_wtchdgd 2>/dev/null)
	bla=$(killall -KILL bat60_trxd 2>/dev/null)
	bla=$(killall -KILL bat60_guid 2>/dev/null)
	#	Lockdateien entfernen, falls diese noch da sind
	rm -f "$DAEMON_LOCK_DIR/bat60_"*
}

###############################################################################
#	@brief		Startet alle Daemons
#
function daemons_on
{
	#TimeDaemon 2>/dev/null 1>/dev/null
	bat60_wtchdgd 2>/dev/null 1>/dev/null
}

function create_logfile
{
	
	#filebase="${0##*/}"
	#filebase="${filebase%.*}"
	
	#fname="$filebase"'_'$(date +%D | sed "s|\/|-|g")".log"
	
	touch "$LOG_FPATH"
	echo "" > "$LOG_FPATH"													#	Log reinigen
}

###############################################################################
#	@brief		log in File mit dem Namen **RUFENDER_FILENAME**.log oder in stdout
#
function bat_log
{
	if [ $is_logopt == true ] ; then
		#filebase="${0##*/}"
		#filebase="${filebase%.*}"
		#fname="$filebase"'_'$(date +%D | sed "s|\/|-|g")".log"
		msg="$1"

		SAVE_IFS=$IFS
		IFS=$'\n'; for tmp_line in $msg; do 
			echo '['$(date)'] : '"$tmp_line" >&1 | tee -a "$LOG_FPATH"
		done
		r_val="${names[*]}"
		IFS=$SAVE_IFS
		
	else
		echo $1
	fi
}

###############################################################################
#	@brief		String von Zeichen reinigen
#
function clean_string
{
	local str=$1
	local syms=$2
	echo $(echo $str | sed "s|[$syms]||g")
}

###############################################################################
#	@brief		String von Zeichen reinigen
#	@param		$1 - Optionen
#
function clean_filename
{
	# Optionen:
	#	p - from Path
	#	x - from eXtention
	
	#ist kein fuehrender Bindestrich, abbrechen
	if [ ${1:0:1} != '-' ] ; then
		bat_log "Error in clean_filename: No leading minus to \$1"
		return 1
	fi

	local idx=1
	local ops=$(clean_string $1 '-')
	local fname=$2
	
	for (( i=0 ; $i<"${#ops}" ; i++ )) ; do
		case "${ops:$i:1}" in
			'p')
				fname="${fname##*/}"
			;;
			'e')
				fname="${fname%.*}"
			;;
		esac
	done
	
	echo $fname
	
	return 0
}



###############################################################################
#	@brief		Findet den Datenträger mit den Updatefiles (USB-Stick, SD-Karte...)
#	@retrun		Absoluter Datenträgerpfad
function find_update_dev
{
	local l_dev=''
	
	# SD-Karte
	l_dev=$(mount | grep $EXT_SD_NAME | awk -F" " '{print ($3)}')
	if [[ $l_dev != "" ]] ; then
		echo $l_dev
		return 0
	fi

	# USB-Stick-A
	l_dev=$(mount | grep $EXT_USB_STICK_A_NAME | awk -F" " '{print ($3)}')
	if [[ $l_dev != "" ]] ; then
		echo $l_dev
		return 0
	fi

	l_dev=$(mount | grep $EXT_USB_STICK_A1_NAME | awk -F" " '{print ($3)}')
	if [[ $l_dev != "" ]] ; then
		echo $l_dev
		return 0
	fi

	# USB-Stick-B
	l_dev=$(mount | grep $EXT_USB_STICK_B_NAME | awk -F" " '{print ($3)}')
	if [[ $l_dev != "" ]] ; then
		echo $l_dev
		return 0
	fi

	l_dev=$(mount | grep $EXT_USB_STICK_B1_NAME | awk -F" " '{print ($3)}')
	if [[ $l_dev != "" ]] ; then
		echo $l_dev
		return 0
	fi

	#if [ -e $EXT_STORAGES_DIR/$EXT_SD_NAME/'update' ] ; then
	#	l_updt_files_dir="$$EXT_STORAGES_DIR/$EXT_SD_NAME/update"
	# USB-Stick-A
	#elif [ -e $EXT_STORAGES_DIR/$EXT_USB_STICK_A_NAME/'update' ] ; then
	#	l_updt_files_dir="$EXT_STORAGES_DIR/$EXT_USB_STICK_A_NAME/update"
	#elif [ -e $EXT_STORAGES_DIR/$EXT_USB_STICK_A1_NAME/'update' ] ; then
	#	l_updt_files_dir="$EXT_STORAGES_DIR/$EXT_USB_STICK_A1_NAME/update"
	# USB-Stick-B
	#elif [ -e $EXT_STORAGES_DIR/$EXT_USB_STICK_B_NAME/'update' ] ; then
	#	l_updt_files_dir="$EXT_STORAGES_DIR/$EXT_USB_STICK_B_NAME/update"
	#elif [ -e $EXT_STORAGES_DIR/$EXT_USB_STICK_B1_NAME/'update' ] ; then
	#	l_updt_files_dir="$EXT_STORAGES_DIR/$EXT_USB_STICK_B1_NAME/update"
	#fi
	
	echo ''
	
	return 1
}

###############################################################################
#	@brief		Findet den absoluten Pfad zur internen SD-Karte
#	@retrun		Absoluter Intern-SD-Kartenpfad
function find_int_sd
{
	local l_sd=''
	
	l_sd=$(mount | grep $INT_SD | awk -F" " '{print ($3)}')
	if [[ $l_sd != "" ]] ; then
		echo $l_sd
		return 0
	fi
}

###############################################################################
#	@brief		Findet den Updatearchiv auf dem Datenträger
#	@retrun		Absoluter Updatefilepfad
function find_update_file
{
	argc=2
	
	#wenn Argumentanzahl nicht uebereinstimmt
	if [ $# -lt $argc ] ;
	then
		echo "find_update_file needs 3 args"
		exit 1
	fi
	
	local l_dir="$1"
	local l_tx_ver="$2"
	
	local REGEX_LS="BAT"$l_tx_ver
	local REGEX_VERS="BAT"$l_tx_ver"_\([0-9]*\).tar.gz"
	
	# local REGEX_VERS="BAT[0-9][0-9]_[a-z][a-z]_\([0-9]*\).tar.gz"
	# local REGEX_VERS_TX="BAT\([0-9][0-9]\)_[a-z][a-z].*"
	
	res=($(ls $l_dir | grep "$REGEX_LS"))
	
	last_ver=0
	last_ver_file=""
	
	#bat_log "Transmitter version is $l_tx_ver"

	for el in "${res[@]}" ; do
		
		local l_tmp_tx_ver=$(echo $el | sed "s/$REGEX_VERS_TX/\1/g")
	
		local ver=$(echo $el | sed "s/$REGEX_VERS/\1/g")
	
		if [ $last_ver -lt $ver ] ; then
			last_ver=$ver
			last_ver_file=$el
		fi
		
	done
	
	#bat_log "File with last version $last_ver_file"

	echo "$l_dir/$last_ver_file"
}

###############################################################################
#	@brief		Findet den Updateordner auf dem Datenträger
#	@retrun		Absoluter Updateodrnerpfad
function find_update_dir
{
	local l_updt_files_dir=''
	local l_mnt_dev=''

	l_mnt_dev=$(find_update_dev)

	if [[ $l_mnt_dev != "" ]] && [ $? -eq 0 ] ; then
		l_updt_files_dir="$l_mnt_dev/$WEATRONIC_FOLFER/update"
		echo $l_updt_files_dir
		return 0
	fi
	
	echo ''
	
	return 1
}

###############################################################################
#	@brief		Gibt die Controller-Adresse nach Namen zurück
#	@param[in]	$1 - Controllerpatternname
#	@retrun		String mit Namen mit ',' getrennt
function get_controller_names
{
	if [ "$#" -lt 1 ] ; then
		echo "Too few args to get_controller_names" 1>&2
		return 1
	fi
	
	local pattrn=$1
	declare -a names
	
	case $pattrn in
		"Transceiver")
			names[0]=$CTRLR_NAME_TRANSCEIVER
		;;
		"HouseKeeper")
			names[0]=$CTRLR_NAME_HOUSEKEEPER
		;;
		"ChargeController")
			names[0]=$CTRLR_NAME_CHARGECTRL
		;;
		"SwitchController")
			names[0]=$CTRLR_NAME_SWITCH_LEFT
			names[1]=$CTRLR_NAME_SWITCH_RIGHT
		;;
		"StickController")
			names[0]=$CTRLR_NAME_STICK_LEFT_EXT
			names[1]=$CTRLR_NAME_STICK_RGHT_EXT
			names[2]=$CTRLR_NAME_STICK_LEFT_INT
			names[3]=$CTRLR_NAME_STICK_RGHT_INT
			#names[2]=$CTRLR_NAME_STICK_LEFT_TOP
			#names[5]=$CTRLR_NAME_STICK_RGHT_TOP
		;;
		*)
			echo ""
			return 1
		;;
	esac
	
	local r_val=''
	
	SAVE_IFS=$IFS
	IFS=","
	r_val="${names[*]}"
	IFS=$SAVE_IFS
	
	echo $r_val
}

###############################################################################
#	@brief		Gibt die Controller-Adresse nach Namen zurück
#	@param[in]	$1 - Controllername
#	@retrun		String mit Addressen mit ', ' getrennt
function get_controller_addr()
{
	if [ "$#" -lt 1 ] ; then
		echo "Too few args to get_controller_addr" 1>&2
		return 1
	fi
	
	local ctrlr=$1
	local addr=''
	
	case $ctrlr in
		"$CTRLR_NAME_TRANSCEIVER")
			addr=$COMM_ADDRESS_TRANSCEIVER
		;;
		"$CTRLR_NAME_HOUSEKEEPER")
			addr=$COMM_ADDRESS_HOUSEKEEPER
		;;
		#	Switch-Panels
		"$CTRLR_NAME_SWITCH_RIGHT")
			addr=$COMM_ADDRESS_SWITCH_RIGHT
		;;
		"$CTRLR_NAME_SWITCH_LEFT")
			addr=$COMM_ADDRESS_SWITCH_LEFT
		;;
		#	Sticks
		"$CTRLR_NAME_STICK_LEFT_EXT")
			addr=$COMM_ADDRESS_STICK_LEFT_EXT
		;;
		"$CTRLR_NAME_STICK_LEFT_INT")
			addr=$COMM_ADDRESS_STICK_LEFT_INT
		;;
		"$CTRLR_NAME_STICK_LEFT_TOP")
			addr=$COMM_ADDRESS_STICK_LEFT_TOP
		;;
		"$CTRLR_NAME_STICK_RGHT_EXT")
			addr=$COMM_ADDRESS_STICK_RGHT_EXT
		;;
		"$CTRLR_NAME_STICK_RGHT_INT")
			addr=$COMM_ADDRESS_STICK_RGHT_INT
		;;
		"$CTRLR_NAME_STICK_RGHT_TOP")
			addr=$COMM_ADDRESS_STICK_RGHT_TOP
		;;
		"$CTRLR_NAME_CHARGECTRL")
			addr=$COMM_ADDRESS_CHARGECTRL
		;;
		*)
			echo ""
			return 1
		;;
	esac
	
	#local r_val=''
	
	#for element in ${addr_arr[@]} ; do
	#	r_val="$r_val""$element, "
	#done
	
	#r_val=${r_val:0:${#r_val}-2}
	
	#echo $r_val
	echo $addr
	
	return 0
}

###############################################################################
#	@brief		Gibt die Controller-Adresse nach Namen zurück
#	@param[in]	$1 - Controllername
#	@retrun		String mit Addressen mit ', ' getrennt
function get_target_name_by_addr()
{
	if [ "$#" -lt 1 ] ; then
		echo "Too few args to get_controller_addr" 1>&2
		return 1
	fi
	
	local addr=$1
	local tgt=''
	
	case $addr in
		"$COMM_ADDRESS_TRANSCEIVER")
			tgt=$CTRLR_NAME_TRANSCEIVER
		;;
		"$COMM_ADDRESS_HOUSEKEEPER")
			tgt=$CTRLR_NAME_HOUSEKEEPER
		;;
		#	Switch-Panels
		"$COMM_ADDRESS_SWITCH_RIGHT")
			tgt=$CTRLR_NAME_SWITCH_RIGHT
		;;
		"$COMM_ADDRESS_SWITCH_LEFT")
			tgt=$CTRLR_NAME_SWITCH_LEFT
		;;
		#	Sticks
		"$COMM_ADDRESS_STICK_LEFT_EXT")
			tgt=$CTRLR_NAME_STICK_LEFT_EXT
		;;
		"$COMM_ADDRESS_STICK_LEFT_INT")
			tgt=$CTRLR_NAME_STICK_LEFT_INT
		;;
		"$COMM_ADDRESS_STICK_LEFT_TOP")
			tgt=$CTRLR_NAME_STICK_LEFT_TOP
		;;
		"$COMM_ADDRESS_STICK_RGHT_EXT")
			tgt=$CTRLR_NAME_STICK_RGHT_EXT
		;;
		"$COMM_ADDRESS_STICK_RGHT_INT")
			tgt=$CTRLR_NAME_STICK_RGHT_INT
		;;
		"$COMM_ADDRESS_STICK_RGHT_TOP")
			tgt=$CTRLR_NAME_STICK_RGHT_TOP
		;;
		"$COMM_ADDRESS_CHARGECTRL")
			tgt=$CTRLR_NAME_CHARGECTRL
		;;
		*)
			echo ""
			return 1
		;;
	esac
	
	echo $tgt
	
	return 0
}

###############################################################################
#	@brief		Distributionsname geben
#	@retrun		Distributionsname
function get_os_distr_name
{
	echo $(lsb_release -a 2>/dev/null | grep 'Distributor ID:' | sed "s|Distributor ID:||g" | sed "s|\s||g")
	
	return 0
}

function get_os_distr_pref
{
	#	tty-Prefix ist Distributionsabhängig
	#	finde die richtige Prefix anhand vom BS-Namen
	local os_name=$(get_os_distr_name)
	local os_pref=''
	
	case $os_name in
		"Ubuntu")
			os_pref='UBUNTU'
		;;
		"MuCross")
			os_pref='LINDIS'
		;;
		*)
			echo ""
			return 1	#	TODO: diverse Rückgabecodes und Bedeutungen
		;;
	esac
	
	echo $os_pref
	
	return 0
}

###############################################################################
#	@brief		Findet die serielle Schnittstelle für einen Controller
#	@param[in]	$1 - Controllername
#	@return		globaler Pfad zur Schnittstelle
function get_isp_iface
{
	if [ "$#" -lt 1 ] ; then
		echo "Too few args to get_isp_iface" 1>&2
		return 1
	fi
	
	local ctrlr=$1
	
	os_pref=$(get_os_distr_pref)
	
	#	Nun kann tty selbst bestimmt werden
	#	Achrung ! die tty's müssen in .conf File definiert werden
	local iface=''
	
	case $ctrlr in
		"$CTRLR_NAME_TRANSCEIVER" | "$CTRLR_NAME_CHARGECTRL"  | "$CTRLR_NAME_SWITCH_LEFT" | "$CTRLR_NAME_SWITCH_RIGHT" | "$CTRLR_NAME_STICK_LEFT_EXT" | "$CTRLR_NAME_STICK_LEFT_INT" | "$CTRLR_NAME_STICK_LEFT_TOP" | "$CTRLR_NAME_STICK_RGHT_EXT" | "$CTRLR_NAME_STICK_RGHT_INT" | "$CTRLR_NAME_STICK_RGHT_TOP")
			iface=${os_pref}'_TRX_UART'
			iface=${!iface}
			
			
		;;
		"$CTRLR_NAME_HOUSEKEEPER")
			iface=${os_pref}'_HSKPR_UART'
			iface=${!iface}
		;;
		*)
			echo ""
			return 1
		;;
	esac
	
	echo $iface
	
	return 0
}

###############################################################################
#	@brief		Findet die serielle Schnittstelle für einen Controller
#	@param[in]	$1 - Controllername
#	@return		globaler Pfad zur Schnittstelle
function get_reinvoke_iface
{
	if [ "$#" -lt 1 ] ; then
		echo "Too few args to get_reinvoke_iface" 1>&2
		return 1
	fi
	
	local ctrlr=$1
	
	local os_pref=$(get_os_distr_pref)
	
	#	Nun kann tty selbst bestimmt werden
	#	Achrung ! die tty's müssen in .conf File definiert werden
	local iface=''
	
	case $ctrlr in
		"$CTRLR_TRANSCEIVER_NAME" | "$CTRLR_NAME_CHARGECTRL"  | "$CTRLR_NAME_SWITCH_LEFT" | "$CTRLR_NAME_SWITCH_RIGHT" | "$CTRLR_NAME_STICK_LEFT_EXT" | "$CTRLR_NAME_STICK_LEFT_INT" | "$CTRLR_NAME_STICK_LEFT_TOP" | "$CTRLR_NAME_STICK_RGHT_EXT" | "$CTRLR_NAME_STICK_RGHT_INT" | "$CTRLR_NAME_STICK_RGHT_TOP" | "$CTRLR_NAME_HOUSEKEEPER")
			iface=${os_pref}'_TRX_UART'
			iface=${!iface}														#	Damit nimmt man den Wert der Variablen, deren Name zusammengestellt wurde	
		;;
		*)
			echo ""
			return 1
		;;
	esac
	
	echo $iface
	
	return 0
}

###############################################################################
#	@brief		Findet die aktuelle Version angegebenes Senderkomponentes
#	@param[in]	$1 - Senderkomponentname
#	@return		Version
function get_current_version
{
	if [ "$#" -lt 1 ] ; then
		echo "Too few args to get_current_version" 1>&2
		return 1
	fi
	
	local comp_name=$1
	
	local current_ver=$(grep "$comp_name" "$BAT_SCRIPTS/includes/tmp/bat_versions" | sed "s|"$comp_name"=||g")
	
	echo $current_ver
	
	return 0
}

###############################################################################
#	@brief		Findet die aktuelle Version angegebenes Senderkomponentes
#	@param[in]	$1 - Senderkomponentname
#	@return		Version
function get_current_ctrlr_version
{
	if [ "$#" -lt 1 ] ; then
		echo "Too few args to get_current_ctrlr_version" 1>&2
		return 1
	fi
	
	local l_comp_name=$1
	local l_target=$(get_controller_addr $l_comp_name)
	local l_iface=$(get_reinvoke_iface $l_comp_name)
	
	l_current_ver=$(bat60_cmd -d $l_iface -t $l_target --execute=getversion -o 0x01 | sed "s|V||g")
	
	echo $l_current_ver
	
	return 0
}

###############################################################################
#	@brief		Controllerupdate-Reihfolgeummer
#	@param[in]	$1 - Controllername
#	@return		Controllerupdate-Reihfolgeummer
function get_ctrlr_order
{
	if [ "$#" -lt 1 ] ; then
		echo "Too few args to get_ctrlr_order" 1>&2
		return 1
	fi
	
	local l_comp_name=$1
	local l_order=''

	declare -a l_order
	
	case $l_comp_name in
		"Transceiver")
			l_order=0
		;;
		"HouseKeeper")
			l_order=1
		;;
		"SwitchController")
			l_order=2
			#l_order[1]=$CTRLR_ORDER_SWITCH_RIGHT
		;;
		"StickController")
			l_order=3
			#l_order[1]=$CTRLR_ORDER_STICK_RGHT_EXT
			#l_order[2]=$CTRLR_ORDER_STICK_LEFT_INT
			#l_order[3]=$CTRLR_ORDER_STICK_RGHT_INT
			#l_order[2]=$CTRLR_ORDER_STICK_LEFT_TOP
			#l_order[5]=$CTRLR_ORDER_STICK_RGHT_TOP
		;;
		*)
			echo ""
			return 1
		;;
	esac

	SAVE_IFS=$IFS
	IFS=","
	l_order="${l_order[*]}"
	IFS=$SAVE_IFS
	
	echo $l_order
	
	return 0
}

###############################################################################
#	@brief		Gibt die Controller-Adresse nach Namen zurück
#	@param[in]	$1 - Controllername
#	@retrun		String mit Addressen mit ', ' getrennt
function get_update_order_by_addr()
{
	if [ "$#" -lt 1 ] ; then
		echo "Too few args to get_controller_addr" 1>&2
		return 1
	fi
	
	local ctrlr=$1
	local l_order=-1
	
	case $ctrlr in
		"$COMM_ADDRESS_TRANSCEIVER")
			l_order=0
		;;
		"$COMM_ADDRESS_HOUSEKEEPER")
			l_order=1
		;;
		#	Switch-Panels
		"$COMM_ADDRESS_SWITCH_RIGHT")
			l_order=3
		;;
		"$COMM_ADDRESS_SWITCH_LEFT")
			l_order=2
		;;
		#	Sticks
		"$COMM_ADDRESS_STICK_LEFT_EXT")
			l_order=4
		;;
		"$COMM_ADDRESS_STICK_LEFT_INT")
			l_order=6
		;;
		"$COMM_ADDRESS_STICK_LEFT_TOP")
			l_order=8
		;;
		"$COMM_ADDRESS_STICK_RGHT_EXT")
			l_order=5
		;;
		"$COMM_ADDRESS_STICK_RGHT_INT")
			l_order=7
		;;
		"$COMM_ADDRESS_STICK_RGHT_TOP")
			l_order=9
		;;
		"$COMM_ADDRESS_CHARGECTRL")
			l_order=10
		;;
		*)
			echo ""
			return 1
		;;
	esac
	
	echo $l_order
	
	return 0
}

###############################################################################
#	@brief		Gibt die Controller-Adresse nach Namen zurück
#	@param[in]	$1 - Controllername
#	@retrun		String mit Addressen mit ', ' getrennt
function get_update_order_by_name()
{
	if [ "$#" -lt 1 ] ; then
		echo "Too few args to get_controller_addr" 1>&2
		return 1
	fi
	
	local ctrlr=$1
	local l_order=-1
	
	case $ctrlr in
		"$CTRLR_NAME_TRANSCEIVER")
			l_order=0
		;;
		"$CTRLR_NAME_HOUSEKEEPER")
			l_order=1
		;;
		#	Switch-Panels
		"$CTRLR_NAME_SWITCH_RIGHT")
			l_order=3
		;;
		"$CTRLR_NAME_SWITCH_LEFT")
			l_order=2
		;;
		#	Sticks
		"$CTRLR_NAME_STICK_LEFT_EXT")
			l_order=4
		;;
		"$CTRLR_NAME_STICK_LEFT_INT")
			l_order=6
		;;
		"$CTRLR_NAME_STICK_LEFT_TOP")
			l_order=8
		;;
		"$CTRLR_NAME_STICK_RGHT_EXT")
			l_order=5
		;;
		"$CTRLR_NAME_STICK_RGHT_INT")
			l_order=7
		;;
		"$CTRLR_NAME_STICK_RGHT_TOP")
			l_order=9
		;;
		"$CTRLR_NAME_CHARGECTRL")
			l_order=10
		;;
		*)
			echo -1
			return 1
		;;
	esac
	
	echo $l_order
	
	return 0
}

###############################################################################
#	@brief		Findet die aktuelle Version angegebenes Senderkomponentes
#	@param[in]	$1 - Senderkomponentname
#	@param[in]	$2 - Version
#	@return		Version
function set_version
{
	if [ "$#" -lt 1 ] ; then
		echo "Too few args to get_tty_iface" 1>&2
		return 1
	fi
	
	local comp_name=$1
	local ver=$2
	
	local current_ver=$(grep "$comp_name" "$BAT_SCRIPTS/includes/tmp/bat_versions")
	
	if [ -z $current_ver ] ; then
		echo "$comp_name=$ver" >> $BAT_SCRIPTS/includes/tmp/bat_versions
	else
		sed -i "s|$comp_name=[a-zA-Z0-9]*|$comp_name=$ver|g" "$BAT_SCRIPTS/includes/tmp/bat_versions"		#	Datei lesen und die LinuxVersion aktualisieren
	fi 
	
	return 0
}

###############################################
# @brief	Entfernt die Datei, falls diese
#			leer ist
#
function rem_if_emty
{
	local l_fname=$1
	local l_file_cont=$(cat $l_fname)
	
	if [[ "" == $l_file_cont ]] ; then
		bat_log 'Removing file '$l_fname
		rm -Rf $l_fname
	fi
}

###############################################
#		@brief	Updatet Controller durch das angegebene Schnittstelle
#		mit der angegebenen Adresse und dem Filenamen
#
function update_target_global
{
	if [ "$#" -lt 4 ] ; then
		echo "Too few args to turn_semitransparency" 1>&2
		return 1
	fi

	local iface=$1
	local addr=$2
	local file=$3
	local l_is_reinvoke=$4
	local update_result=0
	local l_reinvoke_param=''
	local l_tmp_fname=$file
	local l_local_fname=$(clean_filename -p $file)
	local l_int_sd=$(find_int_sd)
	
	if [ -z "$iface" ] ; then
		bat_log "Fatal error: no Iface"
		exit 1
	fi
	
	if [ -z "$addr" ] ; then
		bat_log "Fatal error: no Address"
		exit 1
	fi

	if [ $l_is_reinvoke == true ] ; then
		l_reinvoke_param='--reinvokeisp'
	fi
	
	bat_log "Trying to copy update file to internal SD-card"
	
	#	Datei kopieren
	if [[ "$l_int_sd" != "" ]] ; then
		mkdir -p "$l_int_sd/update_tmp"
		cp -Rf "$file" "$l_int_sd/update_tmp"
		l_tmp_fname="$l_int_sd/update_tmp/$l_local_fname"
		bat_log "File copied to intern SD-card"
	else
		l_tmp_fname="$file"
		bat_log "Error while coping to internal SD-card. Updating from initial storage"
	fi
	
	bat_log "Updating target $addr through $iface"
	
	#	Jede ausgegebene Zeile wird sofort in Log geschieben	
	while read -r update_opt ; do
        if [[ $update_opt == *"ISP_EXIT_STATUS="* ]] ; then
			update_result=$(echo $update_opt | sed "s|ISP_EXIT_STATUS=||g")
        fi

        bat_log "$update_opt"
    done < <(bat60_isp -d $iface --writebootloader=$addr,$l_tmp_fname $l_reinvoke_param -v 2>&1 ;  echo "ISP_EXIT_STATUS=$?" ) 

    bat_log "Update result $update_result"
    
	if [[ $l_int_sd != "" ]] ; then
		rm -Rf "$l_int_sd/update_tmp"
	fi
	
	return $update_result
	
}

###############################################
#		@brief	Senderversion beim HK abrfagen
#
function get_transmitter_variant
{
	echo $("$DAEMON_DIR/bat60_cmd" --execute=transmittervariant -d "$LINDIS_TRX_UART" -t $COMM_ADDRESS_HOUSEKEEPER)
}

###############################################
#		@brief	Setzt die u-boot Spezialvariable, die den Linuxbootstatus abbildet
#
function set_uboot_bootmode
{
	bootmode=$1
	
	if [ $bootmode = "" ] ; then
		bat_log "Error: empty uboot-var"
		return 1
	fi
	
	fw_setenv custom_boot_mode "$bootmode"
}

###############################################
#	@brief	Gibt die u-boot Spezialvariable, die den Linuxbootstatus abbildet
#
function get_uboot_bootmode
{
	bootmode=$(fw_printenv -n custom_boot_mode)
	echo "$bootmode"
}

###############################################
#	@brief	Setzt den Updatevorgagns-Balken auf gegebenen Wert
#	@param	$1 - Prozentwert
#
function set_progress_bar
{
	#wenn Argumentanzahl nicht uebereinstimmt
	if [ $# -lt 1 ] ;
	then
		bat_log "$USAGE_STRING"
		exit 1
	fi
	
	/usr/bin/psplash-write "PROGRESS $1"
}

###############################################
#	@brief	Setzt den Updatevorgagns-Balken auf gegebenen Wert
#	@param	$1 - Nachricht
#
function print_update_msg
{
	#wenn Argumentanzahl nicht uebereinstimmt
	if [ $# -lt 1 ] ;
	then
		bat_log "$USAGE_STRING"
		exit 1
	fi
	
	/usr/bin/psplash-write "MSG $1"
}

###############################################
#	@brief	Initialisiert den Updatevorgagns-Screen
#
function init_update_screen
{
	/usr/bin/psplash-write "QUIT"
	/usr/bin/psplash -a 0 -b /boot/bat60boot.ppm -p 0f01fe -c 050706 -t ffffff &
}

###############################################
#	@brief	Beendet den Updatevorgagns-Screen
#
function destroy_update_screen
{
	/usr/bin/psplash-write "QUIT"
}
